// public/js/chat-widget.js

document.addEventListener('DOMContentLoaded', function () {
    const root = document.getElementById('ip-chat-widget');
    if (!root) return;

    const condominioId = root.dataset.condominioId;
    const residenteId  = root.dataset.residenteId;

    if (!condominioId || !residenteId) return;

    let conversationId         = null;
    let pollMessages           = null;
    let pollUnread             = null;
    let pollInfo               = null;
    let adminOnline            = null;   // true / false
    let lastConversationStatus = null;   // 'open' | 'closed' | null

    // =============================
    //  BOTÓN FLOTANTE
    // =============================
    const btn = document.createElement('button');
    btn.style.position        = 'fixed';
    btn.style.bottom          = '50px';
    btn.style.right           = '40px';
    btn.style.zIndex          = '9999';
    btn.style.borderRadius    = '999px';
    btn.style.padding         = '10px';
    btn.style.border          = 'none';
    btn.style.background      = '#e3e3e3';
    btn.style.boxShadow       = '0 2px 6px rgba(0,0,0,.3)';
    btn.style.cursor          = 'pointer';
    btn.style.display         = 'inline-flex';
    btn.style.alignItems      = 'center';
    btn.style.justifyContent  = 'center';
    btn.style.gap             = '6px';

    // Icono del botón
    const icon = document.createElement('img');
    icon.src         = 'js/chat.png'; // ruta relativa al sitio
    icon.alt         = 'Chat';
    icon.style.width = '45px';
    icon.style.height= '45px';
    icon.style.display = 'block';

    // Punto rojo (notificación)
    const dot = document.createElement('span');
    dot.style.width        = '9px';
    dot.style.height       = '9px';
    dot.style.borderRadius = '50%';
    dot.style.background   = '#dc3545';
    dot.style.display      = 'none';
    dot.style.boxShadow    = '0 0 0 2px #fff';
    dot.style.marginLeft   = '4px';

    btn.appendChild(icon);
    btn.appendChild(dot);

    // =============================
    //  VENTANA DEL CHAT
    // =============================
    const box = document.createElement('div');
    box.style.position      = 'fixed';
    box.style.bottom        = '70px';
    box.style.right         = '20px';
    box.style.width         = '320px';
    box.style.height        = '420px';
    box.style.background    = '#fff';
    box.style.borderRadius  = '10px';
    box.style.boxShadow     = '0 2px 12px rgba(0,0,0,.25)';
    box.style.display       = 'none';
    box.style.zIndex        = '9999';
    box.style.overflow      = 'hidden';
    box.style.fontFamily    = 'Urbanist, system-ui, sans-serif';
    box.style.display       = 'flex';
    box.style.flexDirection = 'column';

    box.innerHTML = `
      <div style="
            background:#4D7EBF;
            color:#fff;
            padding:8px 10px;
            font-size:13px;
            display:flex;
            align-items:center;
            flex:0 0 auto;
        ">
          <span>Soporte</span>

          <!-- Píldora de estado -->
          <span id="ip-chat-status"
                class="widget-status offline"
                style="margin-left:auto;margin-right:8px;">
              offline
          </span>

          <span id="ip-chat-close"
                style="cursor:pointer;font-weight:bold;padding:0 4px;">
              ×
          </span>
      </div>

      <div id="ip-chat-messages"
           style="padding:8px;flex:1 1 auto;overflow-y:auto;background:#f5f5f5;font-size:13px;">
          <p style="color:#777;">Escribe tu mensaje para contactar al administrador.</p>
      </div>

      <form id="ip-chat-form"
            style="border-top:1px solid #ddd;flex:0 0 auto;display:flex;flex-direction:column;padding:6px;">
          
          <textarea id="ip-chat-input"
                    rows="2"
                    style="resize:none;font-size:13px;padding:6px;border-radius:6px;border:1px solid #ccc;max-height:80px;"></textarea>

          <div style="display:flex;justify-content:flex-end;align-items:center;margin-top:6px;gap:8px;">
              <!-- Botón Archivo (izquierda) -->
              <button type="button"
                      id="ip-chat-upload-btn"
                      style="border:none;border-radius:999px;padding:5px 10px;
                             font-size:13px;cursor:pointer;
                             background:#f0f0f0;color:#555;display:inline-flex;
                             align-items:center;gap:6px;">
                  <span style="font-size:12px;">Adjuntar archivo</span>
              </button>

              <!-- Botón Enviar (derecha) -->
              <button type="submit"
                      id="ip-chat-send-btn"
                      style="padding:5px 16px;font-size:12px;border:none;border-radius:999px;
                             background:#007bff;color:#fff;cursor:pointer;
                             display:inline-flex;align-items:center;gap:6px;">
                  <span>Enviar</span> 
              </button>
          </div>

          <!-- input file oculto -->
          <input type="file" id="ip-chat-file" style="display:none;">
      </form>
    `;

    document.body.appendChild(btn);
    document.body.appendChild(box);

    // Referencias DOM
    const closeBtn   = box.querySelector('#ip-chat-close');
    const messagesEl = box.querySelector('#ip-chat-messages');
    const form       = box.querySelector('#ip-chat-form');
    const input      = box.querySelector('#ip-chat-input');
    const fileInput  = box.querySelector('#ip-chat-file');
    const statusEl   = box.querySelector('#ip-chat-status');
    const uploadBtn  = box.querySelector('#ip-chat-upload-btn');

    // Botón Archivo → abre selector
    if (uploadBtn && fileInput) {
        uploadBtn.addEventListener('click', function () {
            fileInput.click();
        });
    }

    // Inicia cerrado
    box.style.display = 'none';

    // =============================
    //   ABRIR / CERRAR WIDGET
    // =============================
    btn.addEventListener('click', function () {
        if (box.style.display === 'none' || box.style.display === '') {
            box.style.display = 'flex';
            initConversation();
        } else {
            box.style.display = 'none';
        }
    });

    closeBtn.addEventListener('click', function () {
        box.style.display = 'none';
    });

    // =============================
    //   ENVÍO FORM (TEXTO + ARCHIVO)
    // =============================
    form.addEventListener('submit', function (e) {
        e.preventDefault();

        const text = input.value.trim();
        const file = fileInput.files[0] || null;

        // 1) Si hay archivo, lo mandamos
        if (file) {
            sendFile(file, text);
            return;
        }

        // 2) Mensaje normal
        if (!text) return;

        if (!conversationId) {
            initConversation(() => sendMessage(text));
            return;
        }

        sendMessage(text);
    });

    // =============================
    //   sendMessage (texto)
    // =============================
    function sendMessage(text) {
        const payload = {
            chat_id: conversationId || 0,
            message: text
        };

        if (!conversationId) {
            payload.condominio_id = condominioId;
            payload.residente_id  = residenteId;
        }

        fetch('chat-resident.php?action=send_message', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        })
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                alert(data.message || 'No se pudo enviar el mensaje');
                return;
            }

            if (!conversationId && data.conversation_id) {
                conversationId         = data.conversation_id;
                lastConversationStatus = 'open';
                startPollingMessages();
                startPollingInfo();
            }

            input.value = '';
            loadMessages();
            dot.style.display = 'none';
        })
        .catch(err => console.error('Error sendMessage', err));
    }

    // =============================
    //   sendFile (archivo)
    // =============================
    function sendFile(file, optionalText) {
        const fd = new FormData();
        fd.append('chat_id',       conversationId || 0);
        fd.append('condominio_id', condominioId);
        fd.append('residente_id',  residenteId);
        fd.append('file',          file);

        fetch('chat-resident.php?action=send_file', {
            method: 'POST',
            body: fd
        })
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                alert(data.message || 'No se pudo enviar el archivo');
                return;
            }

            if (!conversationId && data.conversation_id) {
                conversationId         = data.conversation_id;
                lastConversationStatus = 'open';
                startPollingMessages();
                startPollingInfo();
            }

            fileInput.value = '';
            input.value     = '';
            loadMessages();
            dot.style.display = 'none';
        })
        .catch(err => console.error('Error sendFile', err));
    }

    // =============================
    //   init_conversation
    // =============================
    function initConversation(cb) {
        fetch('chat-resident.php?action=init_conversation', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                condominio_id: condominioId,
                residente_id:  residenteId
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                messagesEl.innerHTML = '<p style="color:red;">No se pudo iniciar el chat.</p>';
                return;
            }

            // Hay conversación abierta → usamos esa
            if (data.conversation_id) {
                conversationId         = data.conversation_id;
                lastConversationStatus = 'open';
                messagesEl.innerHTML   = '';
                loadMessages();
                startPollingMessages();
                startPollingInfo();
                if (cb) cb();
                return;
            }

            // No hay conversación abierta
            conversationId         = null;
            lastConversationStatus = data.last_status || null;
            renderNoConversationMessage();
            input.disabled    = false;
            input.placeholder = 'Escribe para iniciar el chat...';
            if (cb) cb();
        })
        .catch(err => {
            console.error('Error en init_conversation', err);
            messagesEl.innerHTML = '<p style="color:red;">No se pudo iniciar el chat.</p>';
        });
    }

    // =============================
    //   Polling mensajes
    // =============================
    function startPollingMessages() {
        if (pollMessages) return;
        pollMessages = setInterval(loadMessages, 5000);
    }

    function loadMessages() {
        if (!conversationId) return;

        fetch('chat-resident.php?action=get_messages&chat_id=' + conversationId)
            .then(r => r.json())
            .then(data => {
                if (data.error) return;

                messagesEl.innerHTML = '';

                if (!data.length) {
                    messagesEl.innerHTML = '<p style="color:#777;">No hay mensajes aún.</p>';
                    return;
                }

                data.forEach(msg => {
                    const wrap = document.createElement('div');
                    wrap.style.marginBottom = '6px';
                    wrap.style.textAlign   = msg.author_type === 'user' ? 'right' : 'left';

                    const bubble = document.createElement('div');
                    bubble.style.display      = 'inline-block';
                    bubble.style.padding      = '4px 8px';
                    bubble.style.borderRadius = '10px';
                    bubble.style.maxWidth     = '80%';
                    bubble.style.fontSize     = '13px';
                    bubble.style.wordBreak    = 'break-word';
                    bubble.style.background   = (msg.author_type === 'user') ? '#d1e7ff' : '#fff';

                    let content = '';

                    // Archivo adjunto
                    if (msg.file_url) {
                        const url  = msg.file_url;
                        const name = msg.file_name || 'Archivo';
                        const isImage = /\.(jpg|jpeg|png|gif)$/i.test(url);

                        if (isImage) {
                            content += `
                                <div style="margin-bottom:4px;">
                                    <a href="${url}" target="_blank">
                                        <img src="${url}"
                                             alt="${name}"
                                             style="max-width:180px;border-radius:6px;">
                                    </a>
                                </div>
                            `;
                        } else {
                            content += `
                                <div style="margin-bottom:4px;">
                                    <a href="${url}" target="_blank">📎 ${name}</a>
                                </div>
                            `;
                        }
                    }

                    if (msg.text) {
                        content += `<div>${msg.text}</div>`;
                    }

                    bubble.innerHTML = content || '&nbsp;';
                    wrap.appendChild(bubble);
                    messagesEl.appendChild(wrap);
                });

                messagesEl.scrollTop = messagesEl.scrollHeight;
                dot.style.display = 'none';
            })
            .catch(() => {});
    }

    // =============================
    //   Polling info (estado chat)
    // =============================
    function startPollingInfo() {
        if (pollInfo) return;
        pollInfo = setInterval(checkConversationInfo, 5000);
    }

    function checkConversationInfo() {
        if (!conversationId) return;

        fetch('chat-resident.php?action=conversation_info&chat_id=' + conversationId)
            .then(r => r.json())
            .then(info => {
                if (info.error) return;

                const st = String(info.status || '').toLowerCase().trim();
                if (st === 'closed' || st === 'cerrado') {
                    resetClosedChatUI();
                }
            })
            .catch(() => {});
    }

    function resetClosedChatUI() {
        conversationId         = null;
        lastConversationStatus = 'closed';
        clearInterval(pollMessages); pollMessages = null;
        clearInterval(pollInfo);     pollInfo     = null;

        renderNoConversationMessage();
        input.disabled    = false;
        input.placeholder = 'Escribe para iniciar un nuevo chat...';
    }

    // =============================
    //   Punto rojo (unread user)
    // =============================
    function checkUnreadUser() {
        const url = `chat-resident.php?action=unread_user_count&condominio_id=${condominioId}&residente_id=${residenteId}`;

        fetch(url)
            .then(r => r.json())
            .then(data => {
                if (data.error) return;
                const total = Number(data.total || 0);
                dot.style.display = (total > 0 && box.style.display === 'none') ? 'inline-block' : 'none';
            })
            .catch(() => {});
    }

    checkUnreadUser();
    pollUnread = setInterval(checkUnreadUser, 7000);

    // =============================
    //   ONLINE / OFFLINE admins
    // =============================
    function startAdminPresencePolling() {
        checkAdminPresence();
        setInterval(checkAdminPresence, 15000);
    }

    function checkAdminPresence() {
        const url = `chat-resident.php?action=admin_status&condominio_id=${condominioId}`;

        fetch(url)
            .then(r => r.json())
            .then(data => {
                if (data.error) return;
                const online = !!data.online;
                if (adminOnline === online) return;
                adminOnline = online;
                updatePresenceUI();
            })
            .catch(() => {});
    }

    function updatePresenceUI() {
        if (statusEl) {
            if (adminOnline) {
                statusEl.textContent = 'online';
                statusEl.classList.remove('offline');
                statusEl.classList.add('online');
            } else {
                statusEl.textContent = 'offline';
                statusEl.classList.remove('online');
                statusEl.classList.add('offline');
            }
        }

        if (!conversationId) {
            renderNoConversationMessage();
        }
    }

    // =============================
    //   Mensaje base sin conversación
    // =============================
    function renderNoConversationMessage() {
        const status = (lastConversationStatus || '').toLowerCase();

        if (status === 'closed' || status === 'cerrado') {
            messagesEl.innerHTML = `
                <div style="background:#fff3cd;border:1px solid #ffeeba;color:#856404;
                            padding:8px;border-radius:6px;font-size:13px;">
                    Este chat fue cerrado por Administración.<br>
                    Puedes escribir para iniciar uno nuevo.
                </div>
            `;
        } else if (adminOnline === false) {
            messagesEl.innerHTML = `
                <div style="background:#f1f1f1;border:1px solid #ddd;color:#333;
                            padding:8px;border-radius:6px;font-size:13px;">
                    Saludos, en este momento no estamos disponibles.<br>
                    Favor enviarnos sus requerimientos por esta vía y estaremos en contacto
                    lo más pronto posible..... gracias.
                </div>
            `;
        } else {
            messagesEl.innerHTML = `
                <p style="color:#777;margin:0;">
                    No hay conversación activa. Escribe tu mensaje para iniciar un chat.
                </p>
            `;
        }
    }

    // Iniciar polling de presencia de admins
    startAdminPresencePolling();
});
