// public/js/chat-admin.js

document.addEventListener('DOMContentLoaded', function () {
    // Filtros y lista
    const condoSelect         = document.getElementById('filterCondo');
    const statusSelect        = document.getElementById('filterStatus');
    const searchInput         = document.getElementById('searchText');
    const conversationList    = document.getElementById('conversationList');

    // Panel de mensajes
    const messageList         = document.getElementById('messageList');
    const currentChatTitle    = document.getElementById('currentChatTitle');
    const currentChatSubtitle = document.getElementById('currentChatSubtitle');
    const closeChatBtn        = document.getElementById('closeChatBtn');
    const messageForm         = document.getElementById('messageForm');
    const messageInput        = document.getElementById('messageInput');
    const sendBtn             = document.getElementById('sendBtn');
    const fileInput           = document.getElementById('adminFileInput');
    const uploadBtn           = document.getElementById('adminUploadBtn');

    // Nuevo chat (panel izquierdo)
    const btnNewChat          = document.getElementById('btnNewChat');
    const newChatForm         = document.getElementById('newChatForm');
    const newChatCondo        = document.getElementById('newChatCondo');
    const newChatResident     = document.getElementById('newChatResident');
    const newChatSubject      = document.getElementById('newChatSubject');
    const newChatMessage      = document.getElementById('newChatMessage');
    const newChatCancelBtn    = document.getElementById('newChatCancelBtn');
    const newChatSaveBtn      = document.getElementById('newChatSaveBtn');

    // ✅ Botón Delete en header derecha
    const deleteClosedBtn = document.getElementById('deleteClosedBtn');

    // ✅ IDs seleccionados (solo cerrados)
    let selectedClosedIds = new Set();
    let lastSelectionTs = 0;     // cuándo fue la última selección
    let isLoadingConvos = false; // evita llamadas paralelas

    let currentChatId         = null;
    let chatIdToOpenAfterLoad = null; // para abrir el chat recién creado

    // ==========================
    // Eventos filtros
    // ==========================
    condoSelect.addEventListener('change', loadConversations);
    statusSelect.addEventListener('change', loadConversations);
    searchInput.addEventListener('keyup', debounce(loadConversations, 400));

    // ==========================
    // Enviar mensaje (admin)
    // ==========================
    messageForm.addEventListener('submit', function (e) {
        e.preventDefault();
        if (!currentChatId) return;

        const text = messageInput.value.trim();
        if (!text) return;

        sendBtn.disabled = true;

        fetch('chat-admin.php?action=send_message', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                chat_id: currentChatId,
                message: text
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                alert(data.message || 'No se pudo enviar el mensaje');
                return;
            }
            messageInput.value = '';
            loadMessages(currentChatId, true);
        })
        .catch(err => {
            console.error('Error enviando mensaje', err);
            alert('No se pudo enviar el mensaje');
        })
        .finally(() => {
            sendBtn.disabled = false;
        });
    });

    // ==========================
    // Cerrar chat
    // ==========================
    closeChatBtn.addEventListener('click', function () {
        if (!currentChatId) return;
        if (!confirm('¿Seguro que quieres cerrar este chat?')) return;

        closeChatBtn.disabled = true;

        fetch('chat-admin.php?action=close_chat', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ chat_id: currentChatId })
        })
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                alert(data.message || 'No se pudo cerrar el chat');
                return;
            }
            loadConversations();
            clearCurrentChatPanel();
        })
        .catch(err => {
            console.error('Error cerrando chat', err);
            alert('No se pudo cerrar el chat');
        })
        .finally(() => {
            closeChatBtn.disabled = false;
        });
    });

    // ==============================
    // ✅ DELETE cerrados seleccionados
    // ==============================
    if (deleteClosedBtn) {
        deleteClosedBtn.addEventListener('click', () => {
            if (selectedClosedIds.size === 0) return;

            if (!confirm(`¿Borrar ${selectedClosedIds.size} chats cerrados? Esto no se puede deshacer.`)) {
                return;
            }

            fetch('chat-admin.php?action=delete_conversations', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({ ids: Array.from(selectedClosedIds) })
            })
            .then(r => r.json())
            .then(d => {
                if (d.error) {
                    alert(d.message || 'No se pudo borrar');
                    return;
                }

                // limpiar selección y refrescar
                selectedClosedIds.clear();
                updateDeleteBtn();
                loadConversations();
                clearCurrentChatPanel();
            })
            .catch(() => alert('Error borrando chats'));
        });
    }

    // ==========================
    // Botón "Crear nuevo chat"
    // ==========================
    btnNewChat.addEventListener('click', function () {
        const isHidden = newChatForm.classList.contains('d-none');
        if (isHidden) {
            // mostrar formulario
            copyCondosToNewChatSelect();
            if (condoSelect.value) {
                newChatCondo.value = condoSelect.value;
                loadResidentsForNewChat(newChatCondo.value);
            } else {
                newChatResident.innerHTML = '<option value="">Seleccione un residente</option>';
            }
            newChatSubject.value = '';
            newChatMessage.value = '';
            newChatForm.classList.remove('d-none');
        } else {
            // ocultar formulario
            newChatForm.classList.add('d-none');
        }
    });

    newChatCancelBtn.addEventListener('click', function () {
        newChatForm.classList.add('d-none');
    });

    newChatCondo.addEventListener('change', function () {
        if (this.value) {
            loadResidentsForNewChat(this.value);
        } else {
            newChatResident.innerHTML = '<option value="">Seleccione un residente</option>';
        }
    });

    newChatSaveBtn.addEventListener('click', function () {
        const condoId    = newChatCondo.value;
        const residentId = newChatResident.value;
        const subject    = newChatSubject.value.trim();
        const message    = newChatMessage.value.trim();

        if (!condoId) {
            alert('Selecciona un condominio');
            return;
        }
        if (!residentId) {
            alert('Selecciona un residente');
            return;
        }

        newChatSaveBtn.disabled = true;

        fetch('chat-admin.php?action=create_conversation', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                condominio_id: condoId,
                residente_id: residentId,
                subject: subject,
                message: message
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                alert(
                    (data.message || 'No se pudo crear el chat') +
                    (data.detail ? '\n\nDetalle técnico: ' + data.detail : '')
                );
                return;
            }

            chatIdToOpenAfterLoad = data.id || null;

            // refrescamos lista y cerramos formulario
            newChatForm.classList.add('d-none');
            loadConversations();
        })
        .catch(err => {
            console.error('Error creando conversación', err);
            alert('No se pudo crear el chat');
        })
        .finally(() => {
            newChatSaveBtn.disabled = false;
        });
    });

    // ==========================
    // Carga inicial + auto-refresh
    // ==========================
    loadCondos();
    loadConversations();

    // Auto-actualizar lista de conversaciones cada 10 segundos
    setInterval(() => {
    // si está cargando, no lances otra
    if (isLoadingConvos) return;

    // si hay seleccionados, NO refresques
    if (selectedClosedIds.size > 0) return;

    // si el usuario seleccionó hace poco, NO refresques
    if (Date.now() - lastSelectionTs < 8000) return;

    loadConversations();
     }, 10000);

// ==========================
// Adjuntar archivo (admin)
// ==========================
if (uploadBtn && fileInput) {
    uploadBtn.addEventListener('click', function () {
        if (!currentChatId) {
            alert('Selecciona primero un chat para adjuntar un archivo.');
            return;
        }
        fileInput.click(); // abre el selector de archivos
    });

    fileInput.addEventListener('change', function () {
        if (!fileInput.files.length) return;

        if (!currentChatId) {
            alert('Selecciona primero un chat para adjuntar un archivo.');
            fileInput.value = '';
            return;
        }

        sendAdminFile(fileInput.files[0]);
    });
}

// 👇 función dentro del mismo scope (ve todas las constantes)
function sendAdminFile(file) {
    if (!currentChatId) return;

    const fd = new FormData();
    fd.append('chat_id', currentChatId);
    fd.append('file', file);

    sendBtn.disabled      = true;
    closeChatBtn.disabled = true;

    fetch('chat-admin.php?action=send_file', {
        method: 'POST',
        body: fd
    })
    .then(r => r.json())
    .then(data => {
        if (data.error) {
            alert(data.message || 'No se pudo enviar el archivo');
            return;
        }

        fileInput.value = '';
        loadMessages(currentChatId, true);
    })
    .catch(err => {
        console.error('Error enviando archivo (admin)', err);
        alert('Error enviando archivo');
    })
    .finally(() => {
        sendBtn.disabled      = false;
        closeChatBtn.disabled = false;
    });
}


    // ==========================
    // Funciones principales
    // ==========================

    function loadCondos() {
        fetch('chat-admin.php?action=list_condos')
            .then(r => r.json())
            .then(data => {
                if (data.error) {
                    console.error('Error list_condos:', data.message);
                    return;
                }

                condoSelect.innerHTML = '<option value="">Todos los condominios</option>';

                (data || []).forEach(c => {
                    const opt = document.createElement('option');
                    opt.value = c.id;
                    opt.textContent = c.name;
                    condoSelect.appendChild(opt);
                });

                copyCondosToNewChatSelect();
            })
            .catch(err => {
                console.error('Error cargando condominios', err);
            });
    }

    function loadConversations() {

    isLoadingConvos = true;

    const params = new URLSearchParams({
        condo_id: condoSelect.value || '',
        status: statusSelect.value || '',
        q: searchInput.value || ''
    });

    conversationList.innerHTML = '<li class="list-group-item text-muted">Cargando conversaciones…</li>';

    fetch('chat-admin.php?action=list_conversations&' + params.toString())
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                conversationList.innerHTML = '<li class="list-group-item text-danger">' +
                    (data.message || 'Error cargando conversaciones') +
                    '</li>';
                return;
            }

            conversationList.innerHTML = '';

            if (!data || !data.length) {
                conversationList.innerHTML = '<li class="list-group-item text-muted">No se encontraron conversaciones.</li>';
                selectedClosedIds.clear();
                updateDeleteBtn();
                return;
            }

            const closedIdsOnScreen = new Set();

            data.forEach(conv => {
                const li = document.createElement('li');
                li.className = 'list-group-item conversation-item';

                li.innerHTML = `
                    <div class="d-flex justify-content-between">
                        <div>
                            <strong>
                                ${escapeHtml(conv.resident_name || 'Residente')}
                                ${conv.has_unread_admin == 1 ? '<span class="badge bg-danger ms-1">Nuevo</span>' : ''}
                            </strong>
                            <div class="small text-muted">
                                ${escapeHtml(conv.residence || '')}
                            </div>
                            <div class="small text-muted">
                                ${escapeHtml(conv.condo_name || '')}
                            </div>
                            <div class="small">
                                ${escapeHtml(conv.last_message_preview || '')}
                            </div>
                        </div>
                        <div class="text-end">
                            <span class="badge bg-${conv.status === 'closed' ? 'secondary' : 'success'} badge-status">
                                ${conv.status === 'closed' ? 'Cerrado' : 'Abierto'}
                            </span>
                            <div class="small text-muted mt-1">
                                ${escapeHtml(conv.updated_at || '')}
                            </div>
                        </div>
                    </div>
                `;

                li.addEventListener('click', function () {
                    Array.from(conversationList.querySelectorAll('.conversation-item'))
                        .forEach(item => item.classList.remove('active'));

                    li.classList.add('active');
                    openConversation(conv);
                });

                // === CHECKBOX solo para cerrados (alineado y robusto) ===
                const isClosed = String(conv.status || '').toLowerCase().trim() === 'closed';

                if (isClosed) {
                    closedIdsOnScreen.add(String(conv.id)); // ✅ IMPORTANTE para no perder selección

                    li.style.display = 'flex';
                    li.style.alignItems = 'flex-start';
                    li.style.gap = '8px';

                    const cb = document.createElement('input');
                    cb.type = 'checkbox';
                    cb.className = 'closed-select form-check-input mt-1';
                    cb.checked = selectedClosedIds.has(String(conv.id));

                    cb.addEventListener('click', (e) => {
                        e.stopPropagation();

                        const idStr = String(conv.id);
                        if (cb.checked) selectedClosedIds.add(idStr);
                        else selectedClosedIds.delete(idStr);

                        updateDeleteBtn();
                        lastSelectionTs = Date.now();
                    });

                    li.prepend(cb);
                    
                   // ✅ NUEVO: fuerza el bloque interno a ocupar el ancho completo
    const body = li.querySelector('.d-flex.justify-content-between');
    if (body) {
        body.style.flex = '1 1 auto';
        body.style.minWidth = '0';
    }
}

                conversationList.appendChild(li);

                if (chatIdToOpenAfterLoad && Number(chatIdToOpenAfterLoad) === Number(conv.id)) {
                    li.classList.add('active');
                    openConversation(conv);
                    chatIdToOpenAfterLoad = null;
                }
            });

            // limpiar seleccionados que ya no estén visibles
            for (const id of Array.from(selectedClosedIds)) {
                if (!closedIdsOnScreen.has(String(id))) {
                    selectedClosedIds.delete(String(id));
                }
            }
            updateDeleteBtn();
        })
        .catch(err => {
            console.error('Error cargando conversaciones', err);
            conversationList.innerHTML = '<li class="list-group-item text-danger">Error cargando conversaciones.</li>';
        })
        .finally(() => {
            isLoadingConvos = false;  // ✅ AQUÍ VA
        });
}


    function openConversation(conv) {
        currentChatId = conv.id;

        currentChatTitle.textContent = conv.resident_name || 'Residente';
        currentChatSubtitle.textContent = [
            conv.condo_name || '',
            conv.residence || ''
        ].filter(Boolean).join(' · ');

        const isClosed = conv.status === 'closed';
        closeChatBtn.disabled = isClosed;
        sendBtn.disabled      = isClosed;
        messageInput.disabled = isClosed;

        if (isClosed) {
            messageInput.placeholder = 'Este chat está cerrado.';
        } else {
            messageInput.placeholder = 'Escribe tu respuesta al residente…';
        }

        loadMessages(currentChatId, true);
    }

  function loadMessages(chatId, scrollToBottom) {
    if (!chatId) return;

    messageList.innerHTML = '<p class="text-muted mb-0">Cargando mensajes…</p>';

    fetch('chat-admin.php?action=get_messages&chat_id=' + encodeURIComponent(chatId) + '&for=admin')
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                messageList.innerHTML = '<p class="text-danger mb-0">' +
                    (data.message || 'Error cargando mensajes') +
                    '</p>';
                return;
            }

            console.log('get_messages (admin) →', data); // 👈 para verificar file_url en consola

            messageList.innerHTML = '';

            if (!data || !data.length) {
                messageList.innerHTML = '<p class="text-muted mb-0">No hay mensajes todavía.</p>';
                return;
            }

            data.forEach(msg => {
                const wrapper = document.createElement('div');
                wrapper.className = 'chat-message ' +
                    (msg.author_type === 'admin' ? 'admin text-end' : 'user text-start');

                // META (nombre + fecha)
                const meta = document.createElement('div');
                meta.className = 'meta mb-1';
                meta.textContent = (msg.author_name || (msg.author_type === 'admin'
                    ? 'Administrador'
                    : 'Residente')) + ' · ' + (msg.created_at || '');
                wrapper.appendChild(meta);

                // BURBUJA
                const bubble = document.createElement('div');
                bubble.className = 'bubble';

                let html = '';

                // 🔹 Si hay archivo adjunto
               // 🔹 Si hay archivo adjunto
if (msg.file_url) {
    const rawUrl = msg.file_url;                 // lo que viene de PHP, ej: "uploads/chat/xxx.jpg"
    const name   = msg.file_name || 'Archivo';

    // 👇 AJUSTA AQUÍ EL BASE PATH DE TU INSTALACIÓN
    // En tu caso de práctica:
    const BASE_PUBLIC = '/clientes/practica/public/';

    let url;

    if (/^https?:\/\//i.test(rawUrl)) {
        // ya viene con http/https → úsalo tal cual
        url = rawUrl;
    } else if (rawUrl.startsWith('/')) {
        // ya viene como ruta absoluta /algo/...
        url = rawUrl;
    } else {
        // viene como "uploads/chat/xxx.jpg" → lo hacemos absoluto
        url = BASE_PUBLIC + rawUrl.replace(/^\/+/, '');
    }

    const isImage = /\.(jpg|jpeg|png|gif|webp)$/i.test(url);

    if (isImage) {
        html += `
            <div style="margin-bottom:4px;">
                <a href="${url}" target="_blank">
                    <img src="${url}"
                         alt="${escapeHtml(name)}"
                         style="max-width:220px;border-radius:6px;">
                </a>
            </div>
        `;
    } else {
        html += `
            <div style="margin-bottom:4px;">
                <a href="${url}" target="_blank">
                    📎 ${escapeHtml(name)}
                </a>
            </div>
        `;
    }
}


                // 🔹 Texto normal
                if (msg.text) {
                    html += `<div>${escapeHtml(msg.text)}</div>`;
                }

                bubble.innerHTML = html || '&nbsp;';
                wrapper.appendChild(bubble);

                messageList.appendChild(wrapper);
            });

            if (scrollToBottom) {
                messageList.scrollTop = messageList.scrollHeight;
            }
        })
        .catch(err => {
            console.error('Error cargando mensajes', err);
            messageList.innerHTML = '<p class="text-danger mb-0">Error cargando mensajes.</p>';
        });
}






    function clearCurrentChatPanel() {
        currentChatId = null;
        currentChatTitle.textContent    = 'Sin chat seleccionado';
        currentChatSubtitle.textContent = '';
        messageList.innerHTML = '<p class="text-muted mb-0">Selecciona un chat del panel izquierdo para ver los mensajes.</p>';
        closeChatBtn.disabled = true;
        sendBtn.disabled      = true;
        messageInput.disabled = true;
        messageInput.value    = '';
        messageInput.placeholder = 'Selecciona un chat…';
    }

    // ==========================
    // Utilidades
    // ==========================

    function updateDeleteBtn() {
    if (!deleteClosedBtn) return;
    const n = selectedClosedIds.size;
    deleteClosedBtn.disabled = n === 0;
    deleteClosedBtn.textContent = n > 0
        ? `Borrar chat (${n})`
        : 'Borrar chat';
}

    function debounce(fn, delay) {
        let timeout = null;
        return function () {
            const args = arguments;
            clearTimeout(timeout);
            timeout = setTimeout(() => fn.apply(null, args), delay);
        };
    }

    function escapeHtml(str) {
        if (!str) return '';
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    function copyCondosToNewChatSelect() {
        // Copia las opciones del filtro principal al select del formulario de nuevo chat
        newChatCondo.innerHTML = '<option value="">Seleccione un condominio</option>';
        Array.from(condoSelect.options).forEach(opt => {
            if (!opt.value) return; // saltar "Todos los condominios"
            const clone = opt.cloneNode(true);
            newChatCondo.appendChild(clone);
        });
    }

    function loadResidentsForNewChat(condoId) {
        newChatResident.innerHTML = '<option value="">Cargando residentes…</option>';

        const params = new URLSearchParams({
            condo_id: condoId
        });

        fetch('chat-admin.php?action=list_residents&' + params.toString())
            .then(r => r.json())
            .then(data => {
                if (data.error) {
                    newChatResident.innerHTML = '<option value="">Error cargando residentes</option>';
                    alert(data.message || 'No se pudieron cargar los residentes');
                    return;
                }

                newChatResident.innerHTML = '<option value="">Seleccione un residente</option>';

                (data || []).forEach(r => {
                    const opt = document.createElement('option');
                    opt.value = r.id;
                    opt.textContent = r.name;
                    newChatResident.appendChild(opt);
                });

                if (!data || !data.length) {
                    const opt = document.createElement('option');
                    opt.value = '';
                    opt.textContent = 'No hay residentes activos para este condominio';
                    newChatResident.appendChild(opt);
                }
            })
            .catch(err => {
                console.error('Error cargando residentes', err);
                newChatResident.innerHTML = '<option value="">Error cargando residentes</option>';
            });
    }
});






