<?php
// public/chat-admin.php

header('Content-Type: application/json; charset=utf-8');

// Conexión existente
require_once 'dbconnection.php'; // aquí se crea $con (mysqli)

// =========================================
// FUNCIONES AUXILIARES
// =========================================

function json_ok($data = array(), $code = 200)
{
    if (function_exists('http_response_code')) {
        http_response_code($code);
    }
    echo json_encode($data);
    exit;
}

function json_error($message, $code = 400, $extra = array())
{
    if (function_exists('http_response_code')) {
        http_response_code($code);
    }
    echo json_encode(array_merge(array(
        'error'   => true,
        'message' => $message,
    ), $extra));
    exit;
}

function read_json_body()
{
    $raw = file_get_contents('php://input');
    if (!$raw) {
        return array();
    }

    $data = json_decode($raw, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return array();
    }
    return $data;
}

// =========================================
// ENRUTADOR
// =========================================

$action = isset($_GET['action']) ? $_GET['action'] : null;

if (!$action) {
    json_error('Acción no especificada', 400);
}

switch ($action) {
    case 'list_condos':
        list_condos($con);
        break;

    case 'list_conversations':
        list_conversations($con);
        break;

    case 'get_messages':
        get_messages($con);
        break;

    case 'send_message':
        send_message($con);
        break;

    case 'close_chat':
        close_chat($con);
        break;

    case 'list_residents':
        list_residents($con);
        break;

    case 'create_conversation':
        create_conversation($con);
        break;
        
    case 'unread_admin_count':
    unread_admin_count($con);
    break;    

    case 'delete_conversations':
    delete_conversations($con);
    break;

    case 'admin_ping':
    admin_ping($con);
    break;
    
    case 'send_file':
        send_file_admin($con);
        break;

    default:
        json_error('Acción no reconocida: ' . $action, 400);
}

// =========================================
// ACCIONES
// =========================================

/**
 * list_condos
 * Devuelve la lista de condominios (solo activos) para el <select>.
 *
 * Tablas/columnas:
 *  - condominios.CondominioID
 *  - condominios.Nombre
 *  - condominios.Estatus
 */
function list_condos($con)
{
    $sql = "
        SELECT CondominioID AS id, Nombre AS name
        FROM condominios
        WHERE Estatus = 'Activo'
        ORDER BY Nombre ASC
    ";

    $result = mysqli_query($con, $sql);
    if (!$result) {
        json_error('Error consultando condominios', 500, array('detail' => mysqli_error($con)));
    }

    $rows = array();
    while ($row = mysqli_fetch_assoc($result)) {
        $rows[] = $row;
    }

    json_ok($rows);
}

/**
 * list_conversations
 * Lista conversaciones filtradas por condominio, estado y texto.
 *
 * Tablas:
 *  - chat_conversations (nueva)
 *  - chat_messages (nueva)
 *  - residentes (Residentesid, Nombres, Apellidos, Residencia, Estatus)
 *  - condominios (CondominioID, Nombre)
 */
function list_conversations($con)
{
    $condoId = isset($_GET['condo_id']) ? intval($_GET['condo_id']) : 0;
    $status  = isset($_GET['status'])   ? $_GET['status']         : '';
    $q       = isset($_GET['q'])        ? trim($_GET['q'])        : '';

    $where   = " WHERE 1=1 ";

    if ($condoId > 0) {
        $where .= " AND c.condominio_id = " . $condoId . " ";
    }

    if ($status !== '') {
        if ($status === 'open' || $status === 'closed') {
            $where .= " AND c.status = '" . mysqli_real_escape_string($con, $status) . "' ";
        }
    }

    // Solo residentes activos (opcional, pero tiene sentido)
    $where .= " AND r.Estatus = 'Activo' ";

    if ($q !== '') {
        $qEsc = mysqli_real_escape_string($con, $q);
        // Búsqueda por nombre completo, residencia o nombre de condominio
        $where .= " AND (
            CONCAT(COALESCE(r.Nombres,''), ' ', COALESCE(r.Apellidos,'')) LIKE '%{$qEsc}%'
            OR r.Residencia LIKE '%{$qEsc}%'
            OR co.Nombre LIKE '%{$qEsc}%'
        ) ";
    }

    $sql = "
    SELECT
        c.id,
        c.status,
        c.updated_at,
        c.has_unread_admin,
        co.Nombre AS condo_name,
        CONCAT(COALESCE(r.Nombres,''), ' ', COALESCE(r.Apellidos,'')) AS resident_name,
        r.Residencia AS residence,
        (
            SELECT m.text
            FROM chat_messages m
            WHERE m.conversation_id = c.id
            ORDER BY m.created_at DESC, m.id DESC
            LIMIT 1
        ) AS last_message_preview
    FROM chat_conversations c
    INNER JOIN residentes  r  ON r.Residentesid  = c.residente_id
    INNER JOIN condominios co ON co.CondominioID = c.condominio_id
    {$where}
    ORDER BY c.updated_at DESC
    LIMIT 100
";


    $result = mysqli_query($con, $sql);
    if (!$result) {
        json_error('Error consultando conversaciones', 500, array('detail' => mysqli_error($con)));
    }

    $rows = array();
    while ($row = mysqli_fetch_assoc($result)) {
        $rows[] = $row;
    }

    json_ok($rows);
}
function create_conversation($con)
{
    $data = read_json_body();

    $condoId    = isset($data['condominio_id']) ? intval($data['condominio_id']) : 0;
    $residentId = isset($data['residente_id'])  ? intval($data['residente_id'])  : 0;
    $subject    = isset($data['subject']) ? trim($data['subject']) : '';
    $message    = isset($data['message']) ? trim($data['message']) : '';

    if ($condoId <= 0 || $residentId <= 0) {
        json_error('condominio_id y residente_id son requeridos', 400);
    }

    // Verificar condominio
    $sqlCondo = "SELECT CondominioID, Nombre FROM condominios WHERE CondominioID = {$condoId} LIMIT 1";
    $resCondo = mysqli_query($con, $sqlCondo);
    if (!$resCondo) {
        json_error('Error consultando condominio', 500, array('detail' => mysqli_error($con)));
    }
    $condo = mysqli_fetch_assoc($resCondo);
    if (!$condo) {
        json_error('Condominio no encontrado', 404);
    }

    // Verificar residente
    $sqlRes = "
        SELECT Residentesid, Nombres, Apellidos
        FROM residentes
        WHERE Residentesid = {$residentId}
        LIMIT 1
    ";
    $resRes = mysqli_query($con, $sqlRes);
    if (!$resRes) {
        json_error('Error consultando residente', 500, array('detail' => mysqli_error($con)));
    }
    $resident = mysqli_fetch_assoc($resRes);
    if (!$resident) {
        json_error('Residente no encontrado', 404);
    }

    // --- INSERT sencillo en chat_conversations ---
    // SOLO usamos columnas que estamos seguros que existen: condominio_id, residente_id, status
    $sqlInsConv = "
        INSERT INTO chat_conversations (condominio_id, residente_id, status)
        VALUES ({$condoId}, {$residentId}, 'open')
    ";

    if (!mysqli_query($con, $sqlInsConv)) {
        json_error('Error creando conversación', 500, array('detail' => mysqli_error($con)));
    }

    $conversationId = mysqli_insert_id($con);

    // --- Mensaje inicial opcional ---
    if ($message !== '') {
        $authorType = 'admin';
        $authorName = 'Administrador';

        $msgEsc  = mysqli_real_escape_string($con, $message);
        $nameEsc = mysqli_real_escape_string($con, $authorName);
        $typeEsc = mysqli_real_escape_string($con, $authorType);

        $sqlInsMsg = "
            INSERT INTO chat_messages (conversation_id, author_type, author_name, text, created_at)
            VALUES ({$conversationId}, '{$typeEsc}', '{$nameEsc}', '{$msgEsc}', NOW())
        ";

        if (!mysqli_query($con, $sqlInsMsg)) {
            json_error('Error insertando mensaje inicial', 500, array('detail' => mysqli_error($con)));
        }
    }

    json_ok(array(
        'success' => true,
        'id'      => $conversationId
    ));
}

/**
 * list_residents
 * Lista residentes activos, opcionalmente filtrando por condominio_id.
 *
 * Tablas:
 *  - residentes (Residentesid, Nombres, Apellidos, Residencia, Estatus, Condominio)
 *  - condominios (CondominioID, Nombre)
 */
function list_residents($con)
{
    $condoId = isset($_GET['condo_id']) ? intval($_GET['condo_id']) : 0;
    $condoName = '';

    // Si viene un condominio seleccionado, buscamos su Nombre
    if ($condoId > 0) {
        $sqlCondo = "
            SELECT Nombre
            FROM condominios
            WHERE CondominioID = {$condoId}
            LIMIT 1
        ";
        $resCondo = mysqli_query($con, $sqlCondo);
        if (!$resCondo) {
            json_error('Error consultando condominio', 500, array('detail' => mysqli_error($con)));
        }
        $rowCondo = mysqli_fetch_assoc($resCondo);
        if ($rowCondo && isset($rowCondo['Nombre'])) {
            $condoName = mysqli_real_escape_string($con, $rowCondo['Nombre']);
        }
    }

    // Armamos el WHERE
    $where = " WHERE 1=1 ";

    // Solo residentes activos (ajusta si usas otros valores)
    $where .= " AND (r.Estatus = 'Activo' OR r.Estatus = '' OR r.Estatus IS NULL) ";

    // Si tenemos nombre de condominio, filtramos por esa cadena
    if ($condoName !== '') {
        $where .= " AND r.Condominio = '{$condoName}' ";
    }

    $sql = "
        SELECT 
            r.Residentesid AS id,
            CONCAT(
                COALESCE(r.Nombres,''), ' ',
                COALESCE(r.Apellidos,''), ' - ',
                COALESCE(r.Residencia,'')
            ) AS name
        FROM residentes r
        {$where}
        ORDER BY r.Nombres, r.Apellidos
    ";

    $result = mysqli_query($con, $sql);
    if (!$result) {
        json_error('Error consultando residentes', 500, array('detail' => mysqli_error($con)));
    }

    $rows = array();
    while ($row = mysqli_fetch_assoc($result)) {
        $rows[] = $row;
    }

    json_ok($rows);
}


/**
 * get_messages
 * Devuelve todos los mensajes de una conversación.
 * Parámetro GET:
 *   - chat_id
 */
function get_messages($con)
{
    $chatId = isset($_GET['chat_id']) ? intval($_GET['chat_id']) : 0;
    if ($chatId <= 0) {
        json_error('chat_id es requerido', 400);
    }

    // Si la llamada viene del panel admin, marcamos como leído
    if (isset($_GET['for']) && $_GET['for'] === 'admin') {
        $sqlRead = "UPDATE chat_conversations 
                    SET has_unread_admin = 0 
                    WHERE id = {$chatId}";
        mysqli_query($con, $sqlRead);
    }

    // Verificamos que exista la conversación
    $sqlConv = "
        SELECT c.*, 
               CONCAT(COALESCE(r.Nombres,''), ' ', COALESCE(r.Apellidos,'')) AS resident_name
        FROM chat_conversations c
        INNER JOIN residentes r ON r.Residentesid = c.residente_id
        WHERE c.id = {$chatId}
        LIMIT 1
    ";
    $resConv = mysqli_query($con, $sqlConv);
    if (!$resConv) {
        json_error('Error consultando conversación', 500, array('detail' => mysqli_error($con)));
    }

    $conv = mysqli_fetch_assoc($resConv);
    if (!$conv) {
        json_error('Conversación no encontrada', 404);
    }

    // ⬇️ AQUÍ añadimos file_url y file_name
    $sql = "
        SELECT 
            id,
            author_type,
            author_name,
            text,
            file_url,
            file_name,
            created_at
        FROM chat_messages
        WHERE conversation_id = {$chatId}
        ORDER BY created_at ASC, id ASC
    ";

    $result = mysqli_query($con, $sql);
    if (!$result) {
        json_error('Error consultando mensajes', 500, array('detail' => mysqli_error($con)));
    }

    $messages = array();
    while ($row = mysqli_fetch_assoc($result)) {
        $messages[] = array(
            'id'          => (int)$row['id'],
            'author_type' => $row['author_type'],
            'author_name' => $row['author_name'],
            'text'        => $row['text'],
            'file_url'    => $row['file_url'],   // 👈 nuevo
            'file_name'   => $row['file_name'],  // 👈 nuevo
            'created_at'  => $row['created_at'],
        );
    }

    json_ok($messages);
}


/**
 * send_message
 * Inserta un mensaje nuevo en una conversación.
 * Body JSON:
 *   - chat_id
 *   - message
 */
function send_message($con)
{
    $data    = read_json_body();
    $chatId  = isset($data['chat_id']) ? intval($data['chat_id']) : 0;
    $message = isset($data['message']) ? trim($data['message']) : '';

    if ($chatId <= 0 || $message === '') {
        json_error('chat_id y message son requeridos', 400);
    }

    // Verificar que exista la conversación y esté abierta
    $sqlConv = "SELECT * FROM chat_conversations WHERE id = {$chatId} LIMIT 1";
    $resConv = mysqli_query($con, $sqlConv);
    if (!$resConv) {
        json_error('Error consultando conversación', 500, array('detail' => mysqli_error($con)));
    }

    $conv = mysqli_fetch_assoc($resConv);
    if (!$conv) {
        json_error('Conversación no encontrada', 404);
    }

    if ($conv['status'] === 'closed') {
        json_error('La conversación está cerrada, no se pueden enviar más mensajes', 400);
    }

    $authorType = 'admin';
    $authorName = 'Administrador'; // luego puedes traer el nombre real del admin logueado

    $msgEsc   = mysqli_real_escape_string($con, $message);
    $nameEsc  = mysqli_real_escape_string($con, $authorName);
    $typeEsc  = mysqli_real_escape_string($con, $authorType);

    // Insertar mensaje
    $sqlIns = "
        INSERT INTO chat_messages (conversation_id, author_type, author_name, text, created_at)
        VALUES ({$chatId}, '{$typeEsc}', '{$nameEsc}', '{$msgEsc}', NOW())
    ";

    if (!mysqli_query($con, $sqlIns)) {
        json_error('Error insertando mensaje', 500, array('detail' => mysqli_error($con)));
    }

    // Cuando el admin escribe:
    // - el residente tiene nuevo mensaje → has_unread_user = 1
    // - para admin queda leído → has_unread_admin = 0
    $sqlUpd = "
        UPDATE chat_conversations
        SET updated_at = NOW(),
            has_unread_user = 1,
            has_unread_admin = 0
        WHERE id = {$chatId}
    ";
    mysqli_query($con, $sqlUpd);

    json_ok(array('success' => true));
}


/**
 * close_chat
 * Marca la conversación como cerrada.
 * Body JSON:
 *   - chat_id
 */
function close_chat($con)
{
    $data   = read_json_body();
    $chatId = isset($data['chat_id']) ? intval($data['chat_id']) : 0;

    if ($chatId <= 0) {
        json_error('chat_id es requerido', 400);
    }

    // 1) Obtener todos los archivos asociados a este chat
    $sqlFiles = "
        SELECT file_url
        FROM chat_messages
        WHERE conversation_id = {$chatId}
          AND file_url IS NOT NULL
          AND file_url <> ''
    ";
    $resFiles = mysqli_query($con, $sqlFiles);
    if ($resFiles) {
        while ($row = mysqli_fetch_assoc($resFiles)) {
            $fileUrl = $row['file_url'];
            // Asumimos que file_url es algo como 'uploads/chat/xxxx.ext'
            $filePath = __DIR__ . '/' . $fileUrl;
            if (is_file($filePath)) {
                @unlink($filePath);
            }
        }
    }

    // 2) Limpiar referencia en BD para que no queden links rotos
    $sqlClear = "
        UPDATE chat_messages
        SET file_url = NULL,
            file_name = NULL
        WHERE conversation_id = {$chatId}
    ";
    mysqli_query($con, $sqlClear);

    // 3) Cerrar la conversación (ajusta flags si quieres)
    $sqlUpd = "
        UPDATE chat_conversations
        SET status = 'closed',
            has_unread_admin = 0,
            has_unread_user  = 1,
            updated_at = NOW()
        WHERE id = {$chatId}
    ";
    if (!mysqli_query($con, $sqlUpd)) {
        json_error('Error cerrando chat', 500, array('detail' => mysqli_error($con)));
    }

    json_ok(array('success' => true));
}


function delete_conversations($con)
{
    $data = read_json_body();
    $ids  = $data['ids'] ?? [];

    if (!is_array($ids) || count($ids) == 0) {
        json_error('ids es requerido', 400);
    }

    $safeIds = array_filter(array_map('intval', $ids), function($v){ return $v > 0; });
    if (!count($safeIds)) json_error('ids inválidos', 400);

    $idList = implode(',', $safeIds);

    // validar solo cerrados
    $res = mysqli_query($con, "SELECT id FROM chat_conversations WHERE id IN ($idList) AND status='closed'");
    if (!$res) json_error('Error validando chats', 500, ['detail'=>mysqli_error($con)]);

    $valid = [];
    while ($r = mysqli_fetch_assoc($res)) $valid[] = (int)$r['id'];

    if (!count($valid)) json_ok(['deleted'=>0,'ids'=>[]]);

    $validList = implode(',', $valid);

    // borrar mensajes primero
    mysqli_query($con, "DELETE FROM chat_messages WHERE conversation_id IN ($validList)");
    if (mysqli_error($con)) json_error('Error borrando mensajes', 500, ['detail'=>mysqli_error($con)]);

    // borrar conversaciones cerradas
    mysqli_query($con, "DELETE FROM chat_conversations WHERE id IN ($validList)");
    if (mysqli_error($con)) json_error('Error borrando conversaciones', 500, ['detail'=>mysqli_error($con)]);

    json_ok(['deleted'=>count($valid),'ids'=>$valid]);
}

function admin_ping($con)
{
    $data   = read_json_body();
    $userId = isset($data['user_id']) ? intval($data['user_id']) : 0;

    if ($userId <= 0) {
        json_error('user_id es requerido para ping', 400);
    }

    $nowTs = time(); // timestamp UNIX

    $sql = "
        UPDATE tb_users
        SET last_activity = {$nowTs}
        WHERE id = {$userId}
        LIMIT 1
    ";

    if (!mysqli_query($con, $sql)) {
        json_error('Error actualizando última actividad', 500, array(
            'detail' => mysqli_error($con)
        ));
    }

    json_ok(array('success' => true));
}
function send_file_admin($con)
{
    // chat_id obligatorio
    $chatId = isset($_POST['chat_id']) ? intval($_POST['chat_id']) : 0;
    if ($chatId <= 0) {
        json_error('chat_id es requerido', 400);
    }

    if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
        json_error('Archivo no recibido o con error', 400);
    }

    // Verificar conversación y que esté abierta
    $sqlConv = "SELECT * FROM chat_conversations WHERE id = {$chatId} LIMIT 1";
    $resConv = mysqli_query($con, $sqlConv);
    if (!$resConv) {
        json_error('Error consultando conversación', 500, array('detail' => mysqli_error($con)));
    }
    $conv = mysqli_fetch_assoc($resConv);
    if (!$conv) {
        json_error('Conversación no encontrada', 404);
    }
    if ($conv['status'] === 'closed') {
        json_error('La conversación está cerrada, no se pueden enviar más archivos', 400);
    }

    // Validar archivo
    $file     = $_FILES['file'];
    $filename = $file['name'];
    $tmpPath  = $file['tmp_name'];
    $size     = $file['size'];

    // Límite (por ejemplo 10 MB)
    if ($size > 10 * 1024 * 1024) {
        json_error('El archivo supera el límite de 10MB', 400);
    }

    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $allowed = array('jpg','jpeg','png','gif','pdf','doc','docx','xls','xlsx','txt');

    if (!in_array($ext, $allowed)) {
        json_error('Tipo de archivo no permitido', 400);
    }

    // Carpeta de destino
    $uploadDir = __DIR__ . '/uploads/chat';
    if (!is_dir($uploadDir)) {
        @mkdir($uploadDir, 0777, true);
    }

    // Nombre único
    $newName  = time() . '_' . mt_rand(1000,9999) . '_' . preg_replace('/[^A-Za-z0-9_\.\-]/', '_', $filename);
    $destPath = $uploadDir . '/' . $newName;

    if (!move_uploaded_file($tmpPath, $destPath)) {
        json_error('No se pudo guardar el archivo en el servidor', 500);
    }

    // URL relativa para la web
    $fileUrl  = 'uploads/chat/' . $newName;
    $fileName = $filename;

    // Insertar mensaje con archivo
    $authorType = 'admin';
    $authorName = 'Administrador';

    $nameEsc = mysqli_real_escape_string($con, $authorName);
    $typeEsc = mysqli_real_escape_string($con, $authorType);
    $urlEsc  = mysqli_real_escape_string($con, $fileUrl);
    $fnEsc   = mysqli_real_escape_string($con, $fileName);

    $sqlIns = "
        INSERT INTO chat_messages
            (conversation_id, author_type, author_name, text, file_url, file_name, created_at)
        VALUES
            ({$chatId}, '{$typeEsc}', '{$nameEsc}', '', '{$urlEsc}', '{$fnEsc}', NOW())
    ";

    if (!mysqli_query($con, $sqlIns)) {
        json_error('Error insertando mensaje con archivo', 500, array('detail' => mysqli_error($con)));
    }

    // Flags: residente tiene nuevo mensaje
    $sqlUpd = "
        UPDATE chat_conversations
        SET updated_at = NOW(),
            has_unread_user = 1,
            has_unread_admin = 0
        WHERE id = {$chatId}
    ";
    mysqli_query($con, $sqlUpd);

    json_ok(array(
        'success' => true,
        'chat_id' => $chatId,
        'file_url' => $fileUrl,
        'file_name' => $fileName
    ));
}
function unread_admin_count($con)
{
    $sql = "
        SELECT COUNT(*) AS total
        FROM chat_conversations
        WHERE status = 'open'
          AND has_unread_admin = 1
    ";

    $res = mysqli_query($con, $sql);
    if (!$res) {
        json_error('Error contando no leídos admin', 500, array('detail' => mysqli_error($con)));
    }

    $row = mysqli_fetch_assoc($res);
    json_ok(array('total' => (int)($row['total'] ?? 0)));
}

